<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;
use yii\web\IdentityInterface;

class User extends ActiveRecord implements IdentityInterface
{

    public $currentPassword;
    public $newPassword;
    public $newPasswordRepeat;

    public static function tableName()
    {
        return '{{%users}}';
    }

    public function rules()
    {
        return [
            [['username', 'name', 'surname', 'email'], 'required', 'on' => 'profile'],
            [['currentPassword', 'newPassword', 'newPasswordRepeat'], 'required', 'on' => 'changePassword'],
            ['newPassword', 'string', 'min' => 6, 'on' => 'changePassword'],
            ['newPasswordRepeat', 'compare', 'compareAttribute' => 'newPassword', 'on' => 'changePassword'],
            [['username', 'name', 'surname', 'email', 'password_hash'], 'required'],
            [['name', 'surname', 'patronymic'], 'string', 'max' => 100],
            [['username'], 'string', 'max' => 100],
            [['email'], 'string', 'max' => 150],
            [['username'], 'unique'],
            [['email'], 'unique'],
            [['auth_key'], 'string', 'max' => 32],
        ];
    }

    public function attributeLabels()
    {
        return [
            'username' => 'Логин',
            'name' => 'Имя',
            'surname' => 'Фамилия',
            'patronymic' => 'Отчество',
            'email' => 'Email',
        ];
    }

    /**
     * {@inheritdoc}
     */
    public static function findIdentity($id)
    {
        return static::findOne(['id' => $id, 'status' => 10]);
    }

    /**
     * {@inheritdoc}
     */
    public static function findIdentityByAccessToken($token, $type = null)
    {
        return null;
    }

    /**
     * {@inheritdoc}
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * {@inheritdoc}
     */
    public function getAuthKey()
    {
        return $this->auth_key;
    }

    /**
     * {@inheritdoc}
     */
    public function validateAuthKey($authKey)
    {
        return $this->auth_key === $authKey;
    }

    public function generateAuthKey()
    {
        $this->auth_key = Yii::$app->security->generateRandomString();
    }

    /**
     * Validates password
     *
     * @param string $password password to validate
     * @return bool if password provided is valid for current user
     */
    public function validatePassword($password)
    {
        return Yii::$app->security->validatePassword($password, $this->password_hash);
    }

    public function setPassword($password)
    {
        $this->password_hash = Yii::$app->security->generatePasswordHash($password);
    }

    public function getFullName()
    {
        return $this->surname . ' ' . $this->name . ($this->patronymic ? ' ' . $this->patronymic : '');
    }

    public static function findByUsername($username)
    {
        return static::find()
            ->where(['or',
                ['username' => $username],
                ['email' => $username]
            ])
            ->andWhere(['status' => 10])
            ->one();
    }

    public function getUnreadNotificationsCount()
    {
        return \app\models\Notification::getUnreadCount($this->id);
    }

    /**
     * Gets query for [[Notifications]].
     *
     * @return \yii\db\ActiveQuery
     */
    public function getNotifications()
    {
        return $this->hasMany(\app\models\Notification::class, ['user_id' => 'id'])
            ->orderBy(['created_at' => SORT_DESC]);
    }

    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            if ($this->isNewRecord) {
                $this->generateAuthKey();
                $time = time();
                $this->created_at = $time;
                $this->updated_at = $time;
            } else {
                $this->updated_at = time();
            }
            return true;
        }
        return false;
    }

    public static function getProfileForm($id = null)
    {
        if ($id === null && !Yii::$app->user->isGuest) {
            $id = Yii::$app->user->id;
        }

        $user = static::findOne($id);
        if (!$user) {
            return null;
        }

        return $user;
    }

    public function rulesProfile()
    {
        return [
            [['name', 'surname', 'username', 'email'], 'required'],
            [['name', 'surname', 'patronymic'], 'match', 'pattern' => '/^[а-яА-Я\s\-]+$/u'],
            ['username', 'match', 'pattern' => '/^[a-zA-Z0-9\-]+$/'],
            ['username', 'unique', 'filter' => ['<>', 'id', $this->id]],
            ['email', 'email'],
            ['email', 'unique', 'filter' => ['<>', 'id', $this->id]],
        ];
    }

    public function saveProfile($data)
    {
        $this->load($data, '');
        return $this->save();
    }

    public function changePassword($currentPassword, $newPassword)
    {
        if (!$this->validatePassword($currentPassword)) {
            $this->addError('currentPassword', 'Текущий пароль неверен.');
            return false;
        }

        $this->setPassword($newPassword);
        return $this->save();
    }
    public function isAdmin()
    {
        return $this->role === 'admin';
    }

    public function isExpert()
    {
        return $this->role === 'expert';
    }

    public function isUser()
    {
        return $this->role === 'user';
    }
}