<?php

namespace app\controllers;

use app\models\Notification;
use app\models\User;
use Yii;
use app\models\ApplicationForm;
use app\models\Applications;
use app\models\Categories;
use app\models\Contests;
use yii\data\ActiveDataProvider;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use yii\filters\AccessControl;

class CabinetController extends Controller
{
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    [
                        'allow' => true,
                        'roles' => ['@'],
                        'matchCallback' => function($rule, $action) {
                            $user = Yii::$app->user->identity;
                            return $user->role === 'user' || $user->role === 'admin';
                        }
                    ],
                ],
                'denyCallback' => function() {
                    Yii::$app->session->setFlash('error', 'Доступ запрещен.');
                    return Yii::$app->response->redirect(['site/index']);
                }
            ],
        ];
    }

    public function actionIndex()
    {
        return $this->render('index', [
            'user' => Yii::$app->user->identity,
        ]);
    }

    public function actionApplications()
    {
        $userId = Yii::$app->user->id;
        $applications = Applications::find()
            ->where(['user_id' => $userId])
            ->orderBy(['created_at' => SORT_DESC])
            ->all();

        return $this->render('applications', [
            'applications' => $applications,
        ]);
    }

    public function actionApplicationView($id)
    {
        $application = Applications::findOne(['id' => $id, 'user_id' => Yii::$app->user->id]);

        if (!$application) {
            throw new NotFoundHttpException('Заявка не найдена.');
        }

        return $this->render('application-view', [
            'application' => $application,
        ]);
    }

    public function actionApplicationCancel($id)
    {
        $application = Applications::findOne(['id' => $id, 'user_id' => Yii::$app->user->id]);

        if (!$application) {
            throw new NotFoundHttpException('Заявка не найдена.');
        }

        $application->status = 'blocked';
        if ($application->save()) {
            Yii::$app->session->setFlash('success', 'Заявка отменена.');
        } else {
            Yii::$app->session->setFlash('error', 'Ошибка при отмене заявки.');
        }

        return $this->redirect(['applications']);
    }

    public function actionApply($contest_id)
    {
        $contest = Contests::findOne($contest_id);

        if (!$contest) {
            throw new NotFoundHttpException('Конкурс не найден.');
        }

        $model = new ApplicationForm();
        $model->contest_id = $contest_id;

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->session->setFlash('success', 'Заявка успешно подана!');
            return $this->redirect(['/site/index']);
        }

        $nominations = Categories::find()
            ->where(['contest_id' => $contest_id, 'type' => 'nomination', 'status' => 1])
            ->all();

        $ageCategories = Categories::find()
            ->where(['contest_id' => $contest_id, 'type' => 'age', 'status' => 1])
            ->all();

        return $this->render('apply', [
            'model' => $model,
            'contest' => $contest,
            'nominations' => $nominations,
            'ageCategories' => $ageCategories,
        ]);
    }

    public function actionProfile()
    {
        $model = User::findOne(Yii::$app->user->id);

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->session->setFlash('success', 'Профиль успешно обновлен!');
            return $this->refresh();
        }

        return $this->render('profile', [
            'model' => $model,
        ]);
    }

    public function actionChangePassword()
    {
        $model = User::findOne(Yii::$app->user->id);

        if (Yii::$app->request->isPost) {
            $currentPassword = Yii::$app->request->post('currentPassword');
            $newPassword = Yii::$app->request->post('newPassword');

            if ($model->changePassword($currentPassword, $newPassword)) {
                Yii::$app->session->setFlash('success', 'Пароль успешно изменен!');
                return $this->refresh();
            }
        }

        return $this->render('change-password', [
            'model' => $model,
        ]);
    }

    public function actionNotifications($page = 1)
    {
        $userId = Yii::$app->user->id;

        $query = Notification::find()
            ->where(['user_id' => $userId])
            ->orderBy(['created_at' => SORT_DESC]);

        $countQuery = clone $query;
        $totalCount = $countQuery->count();
        $pageSize = 10;

        $pagination = new \yii\data\Pagination([
            'totalCount' => $totalCount,
            'pageSize' => $pageSize,
            'page' => $page - 1,
        ]);

        $notifications = $query
            ->offset($pagination->offset)
            ->limit($pagination->limit)
            ->all();

        return $this->render('notifications', [
            'notifications' => $notifications,
            'pagination' => $pagination,
        ]);
    }

    public function actionMarkAsRead($id)
    {
        $notification = Notification::findOne([
            'id' => $id,
            'user_id' => Yii::$app->user->id
        ]);

        if ($notification && $notification->markAsRead()) {
            Yii::$app->session->setFlash('success', 'Уведомление отмечено как прочитанное');
        } else {
            Yii::$app->session->setFlash('error', 'Не удалось отметить уведомление');
        }

        return $this->redirect(['notifications']);
    }

    public function actionMarkAllAsRead()
    {
        $userId = Yii::$app->user->id;

        Notification::updateAll(
            ['is_read' => Notification::STATUS_READ],
            ['user_id' => $userId, 'is_read' => Notification::STATUS_UNREAD]
        );

        Yii::$app->session->setFlash('success', 'Все уведомления отмечены как прочитанные');
        return $this->redirect(['notifications']);
    }

    public function actionGetUnreadCount()
    {
        if (Yii::$app->user->isGuest) {
            return $this->asJson(['count' => 0]);
        }

        $count = Yii::$app->user->identity->getUnreadNotificationsCount();
        return $this->asJson(['count' => $count]);
    }
}




