<?php
namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;
use yii\web\NotFoundHttpException;
use app\models\Contests;
use app\models\Categories;
use app\models\Applications;
use app\models\User;
use app\models\Notification;
use app\models\ExpertAssignments;

class AdminController extends Controller
{
    public $layout = 'admin';

    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    [
                        'allow' => true,
                        'matchCallback' => function($rule, $action) {
                            if (Yii::$app->user->isGuest) return false;
                            return Yii::$app->user->identity->role === 'admin';
                        }
                    ],
                ],
                'denyCallback' => function() {
                    return Yii::$app->response->redirect(['site/index']);
                }
            ],
        ];
    }

    public function actionIndex()
    {
        $stats = [
            'contests' => Contests::find()->count(),
            'applications' => Applications::find()->count(),
            'users' => User::find()->count(),
            'experts' => User::find()->where(['role' => 'expert'])->count(),
        ];

        return $this->render('index', ['stats' => $stats]);
    }

    public function actionContests()
    {
        $contests = Contests::find()->orderBy(['created_at' => SORT_DESC])->all();
        return $this->render('contests', ['contests' => $contests]);
    }

    public function actionContestCreate()
    {
        $model = new Contests();

        if (Yii::$app->request->isPost) {
            $model->load(Yii::$app->request->post());
            $model->created_at = time();
            $model->updated_at = time();

            if ($model->save()) {
                Yii::$app->session->setFlash('success', 'Конкурс создан');
                return $this->redirect(['contests']);
            }
        }

        return $this->render('contest-form', ['model' => $model]);
    }

    public function actionContestUpdate($id)
    {
        $model = Contests::findOne($id);
        if (!$model) throw new NotFoundHttpException('Конкурс не найден');

        if (Yii::$app->request->isPost) {
            $model->load(Yii::$app->request->post());
            $model->updated_at = time();

            if ($model->save()) {
                Yii::$app->session->setFlash('success', 'Конкрс обновлен');
                return $this->redirect(['contests']);
            }
        }

        return $this->render('contest-form', ['model' => $model]);
    }

    public function actionContestDelete($id)
    {
        $model = Contests::findOne($id);
        if ($model) {
            $model->delete();
            Yii::$app->session->setFlash('success', 'Конкурс удалун');
        }
        return $this->redirect(['contests']);
    }

    public function actionCategories($type = 'nomination')
    {
        $categories = Categories::find()
            ->where(['type' => $type])
            ->orderBy(['created_at' => SORT_DESC])
            ->all();

        return $this->render('categories', [
            'categories' => $categories,
            'type' => $type
        ]);
    }

    public function actionCategoryCreate($type)
    {
        $model = new Categories();
        $model->type = $type;

        if (Yii::$app->request->isPost) {
            $model->load(Yii::$app->request->post());
            $model->created_at = time();

            if ($model->save()) {
                Yii::$app->session->setFlash('success', 'Категория создана');
                return $this->redirect(['categories', 'type' => $type]);
            }
        }

        $contests = Contests::find()->all();
        return $this->render('category-form', [
            'model' => $model,
            'contests' => $contests,
            'type' => $type
        ]);
    }

    public function actionApplications()
    {
        $search = Yii::$app->request->get('search');
        $applications = Applications::find()
            ->joinWith(['user'])
            ->orderBy(['created_at' => SORT_DESC]);

        if ($search) {
            $applications->andWhere(['like', 'user.surname', $search]);
        }

        $applications = $applications->all();

        return $this->render('applications', [
            'applications' => $applications,
            'search' => $search
        ]);
    }

    public function actionApplicationView($id)
    {
        $application = Applications::findOne($id);

        return $this->render('application-view', ['application' => $application]);
    }

    public function actionApplicationBlock($id)
    {
        $application = Applications::findOne($id);
        if ($application) {
            $application->status = 'blocked';
            $application->save(false);
            Yii::$app->session->setFlash('success', 'Заявка заблокирована');
        }
        return $this->redirect(['applications']);
    }

    public function actionUsers()
    {
        $users = User::find()->orderBy(['created_at' => SORT_DESC])->all();
        return $this->render('users', ['users' => $users]);
    }

    public function actionUserCreate()
    {
        $model = new User();
        $model->scenario = 'create';

        if (Yii::$app->request->isPost) {
            $post = Yii::$app->request->post();
            $model->load($post);
            $model->setPassword($post['User']['password']);
            $model->status = 10;

            if ($model->save()) {
                return $this->redirect(['users']);
            }
        }

        return $this->render('user-form', ['model' => $model]);
    }

    public function actionExperts()
    {
        $experts = User::find()->where(['role' => 'expert'])->all();
        return $this->render('experts', ['experts' => $experts]);
    }

    public function actionAssignExpert($expert_id)
    {
        $model = new ExpertAssignments();
        $expert = User::findOne($expert_id);

        if (Yii::$app->request->isPost) {
            $model->load(Yii::$app->request->post());
            $model->expert_id = $expert_id;
            $model->created_at = time();

            if ($model->save()) {
                Yii::$app->session->setFlash('success', 'Эксперт назначен');
                return $this->redirect(['experts']);
            }
        }

        $contests = Contests::find()->all();
        $nominations = Categories::find()->where(['type' => 'nomination'])->all();
        $ageCategories = Categories::find()->where(['type' => 'age'])->all();

        return $this->render('assign-expert', [
            'model' => $model,
            'expert' => $expert,
            'contests' => $contests,
            'nominations' => $nominations,
            'ageCategories' => $ageCategories
        ]);
    }

    public function actionNotifications()
    {
        $notifications = Notification::find()
            ->orderBy(['created_at' => SORT_DESC])
            ->all();

        return $this->render('notifications', ['notifications' => $notifications]);
    }

    public function actionNotificationCreate()
    {
        $model = new Notification();

        if (Yii::$app->request->isPost) {
            $model->load(Yii::$app->request->post());
            $model->is_read = 0;
            $model->created_at = time();

            if ($model->save()) {
                Yii::$app->session->setFlash('success', 'Уведомлние отправлено');
                return $this->redirect(['notifications']);
            }
        }

        $users = User::find()->all();
        return $this->render('notification-form', [
            'model' => $model,
            'users' => $users
        ]);
    }

    public function actionCategoryUpdate($id)
    {
        $model = Categories::findOne($id);
        if (!$model) throw new NotFoundHttpException('Категория не найдена');

        if (Yii::$app->request->isPost) {
            $model->load(Yii::$app->request->post());
            if ($model->save()) {
                Yii::$app->session->setFlash('success', 'Категория обновлена');
                return $this->redirect(['categories', 'type' => $model->type]);
            }
        }

        $contests = Contests::find()->all();
        return $this->render('category-form', [
            'model' => $model,
            'contests' => $contests,
            'type' => $model->type
        ]);
    }

    public function actionCategoryDelete($id)
    {
        $model = Categories::findOne($id);
        if ($model) {
            $type = $model->type;
            $model->delete();
            Yii::$app->session->setFlash('success', 'Категория была удалена');
        }
        return $this->redirect(['categories', 'type' => $type]);
    }
}